import React, { useState, useEffect, useRef } from 'react';
import { 
  ShoppingCart, Check, Star, Shield, Zap, ArrowRight, TrendingUp, 
  Cpu, Lock, Globe, DollarSign, Activity, Save, AlertCircle,
  Settings, Monitor, LayoutDashboard, LogOut, Menu, X, Users, CreditCard,
  BarChart3, ChevronRight, List, Mail, FileText, Download, Upload, MessageSquare
} from 'lucide-react';

// --- CONFIGURATION STATE (DATABASE) ---
const initialSiteConfig = {
  general: {
    siteName: "QUANTUM.AI",
    primaryColor: "green", 
  },
  topBar: {
    isEnabled: true,
    text: "🔥 FLASH SALE ACTIVE:",
    highlightText: "80% OFF ENDS SOON",
    timerMinutes: 15
  },
  hero: {
    headlinePrefix: "Automate Your",
    headlineSuffix: "Trading Income.",
    subHeadline: "The world's first AI-driven scalping bot for MT4/MT5. Proven 94% win-rate on Gold & Bitcoin pairs.",
    ctaText: "GET INSTANT ACCESS",
    showChart: true
  },
  product: {
    price: 197,
    regularPrice: 997,
    currency: "$",
    discountBadge: "SAVE $800 TODAY"
  },
  downloads: {
    fileLink: "https://example.com/download/quantum-bot-v4.zip", // Admin can change this
    version: "v4.2.1 (Latest)"
  },
  aboutUs: {
    content: "QuantumAI was founded in 2021 by a team of ex-Wall Street quant traders and machine learning experts. Our mission is to democratize institutional-grade trading algorithms for retail traders. We believe that financial freedom should be accessible to everyone, not just hedge funds."
  },
  stats: {
    isEnabled: true,
    totalProfit: "$4,281,092",
    activeUsers: "1,402",
    dailyWinRate: "94.2%"
  },
  ticker: {
    isEnabled: true,
    text: "🟢 NEW TRADE: BTC/USD Buy @ 64,200 | 🟢 TP HIT: XAU/USD +45 Pips | 🔴 SL HIT: EUR/USD -10 Pips | 🟢 NEW USER: Alex just joined from UK..."
  }
};

// --- HELPER COMPONENTS ---

const LiveChart = ({ color }) => {
  const [data, setData] = useState(Array(60).fill(1950));
  const [currentPrice, setCurrentPrice] = useState(1950.00);
  const [trend, setTrend] = useState('up');
  
  useEffect(() => {
    const interval = setInterval(() => {
      setData(prev => {
        const last = prev[prev.length - 1];
        const volatility = Math.random() > 0.9 ? 5 : 1;
        const direction = Math.random() > 0.45 ? 1 : -1;
        const move = (Math.random() * volatility * direction);
        let next = last + move;
        setCurrentPrice(next);
        setTrend(move >= 0 ? 'up' : 'down');
        return [...prev.slice(1), next];
      });
    }, 800); 
    return () => clearInterval(interval);
  }, []);

  const getStrokeColor = () => {
    switch(color) {
      case 'blue': return '#3b82f6';
      case 'purple': return '#a855f7';
      case 'yellow': return '#eab308';
      default: return '#22c55e';
    }
  };

  const min = Math.min(...data);
  const max = Math.max(...data);
  const range = max - min || 1;
  const points = data.map((d, i) => `${i * (100/(data.length-1))},${100 - ((d - min) / range * 80 + 10)}`).join(' ');

  return (
    <div className="bg-slate-900 border border-slate-700/50 rounded-xl overflow-hidden shadow-2xl backdrop-blur-sm relative group h-[320px] md:h-[400px] w-full max-w-5xl mx-auto mt-12">
      <div className="flex justify-between items-center p-4 border-b border-white/5 bg-slate-900/50">
        <div className="flex items-center gap-3">
          <div className="flex gap-1.5">
            <div className="w-2.5 h-2.5 rounded-full bg-red-500"></div>
            <div className="w-2.5 h-2.5 rounded-full bg-yellow-500"></div>
            <div className="w-2.5 h-2.5 rounded-full bg-green-500"></div>
          </div>
          <span className="text-xs font-mono text-slate-400">XAUUSD [M1] • LIVE MARKET DATA</span>
        </div>
        <div className={`font-mono font-bold text-lg ${trend === 'up' ? 'text-green-500' : 'text-red-500'} transition-colors duration-500`}>
          {currentPrice.toFixed(2)}
        </div>
      </div>
      <div className="relative h-full w-full p-4">
        <div className="absolute inset-0 bg-[linear-gradient(rgba(255,255,255,0.02)_1px,transparent_1px),linear-gradient(90deg,rgba(255,255,255,0.02)_1px,transparent_1px)] bg-[size:40px_40px]"></div>
        <svg className="w-full h-full overflow-visible" preserveAspectRatio="none" viewBox="0 0 100 100">
           <defs>
             <linearGradient id="chartFill" x1="0" y1="0" x2="0" y2="1">
               <stop offset="0%" stopColor={getStrokeColor()} stopOpacity="0.2" />
               <stop offset="100%" stopColor={getStrokeColor()} stopOpacity="0" />
             </linearGradient>
           </defs>
           <path d={`M 0,100 L 0,${100 - ((data[0] - min) / range * 80 + 10)} ${points} V 100 Z`} fill="url(#chartFill)" />
           <polyline points={points} fill="none" stroke={getStrokeColor()} strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" vectorEffect="non-scaling-stroke" />
           <circle cx="100" cy={100 - ((data[data.length-1] - min) / range * 80 + 10)} r="2" fill="white" className="animate-ping" />
           <circle cx="100" cy={100 - ((data[data.length-1] - min) / range * 80 + 10)} r="2" fill={getStrokeColor()} />
        </svg>
      </div>
    </div>
  );
};

// 2. CHECKOUT MODAL
const CheckoutModal = ({ config, onClose, onOrderPlaced }) => {
  const [isProcessing, setIsProcessing] = useState(false);
  const [paymentMethod, setPaymentMethod] = useState('card');
  const [formData, setFormData] = useState({
    firstName: '',
    lastName: '',
    email: '',
    country: 'United States',
    cardName: '',
    cardNumber: '',
    expiry: '',
    cvc: ''
  });

  const handleChange = (e) => {
    setFormData({ ...formData, [e.target.name]: e.target.value });
  };

  const handlePay = (e) => {
    e.preventDefault();
    setIsProcessing(true);
    
    setTimeout(() => {
      setIsProcessing(false);
      
      const newOrder = {
        id: Math.floor(Math.random() * 100000),
        firstName: formData.firstName,
        lastName: formData.lastName,
        email: formData.email,
        country: formData.country,
        amount: config.product.price,
        date: new Date().toISOString(),
        method: paymentMethod,
        status: 'Paid'
      };
      
      onOrderPlaced(newOrder); 
      onClose();
    }, 2000);
  };

  return (
    <div className="fixed inset-0 z-[100] flex items-end sm:items-center justify-center sm:p-4 bg-black/80 backdrop-blur-sm animate-fade-in font-sans">
       <div className="bg-white text-slate-900 w-full sm:max-w-lg rounded-t-2xl sm:rounded-2xl overflow-hidden shadow-2xl relative animate-scale-in max-h-[90vh] flex flex-col">
          <button onClick={onClose} className="absolute top-4 right-4 text-slate-400 hover:text-slate-600 transition z-10"><X size={24}/></button>
          <div className="bg-slate-50 p-6 border-b border-slate-100 flex justify-between items-center relative shrink-0">
             <div><h3 className="font-bold text-xl text-slate-900">Secure Checkout</h3></div>
             <div className="text-right">
               <div className="text-xs text-slate-500 uppercase font-bold">Total</div>
               <div className="text-2xl font-black text-slate-900">{config.product.currency}{config.product.price}</div>
             </div>
          </div>
          <div className="p-6 overflow-y-auto">
             <form onSubmit={handlePay} className="space-y-4">
                <div className="grid grid-cols-2 gap-4">
                   <div><label className="block text-[11px] font-bold uppercase text-slate-500 mb-1.5">First Name</label><input required name="firstName" value={formData.firstName} onChange={handleChange} type="text" className="w-full bg-white border border-slate-300 rounded-lg p-3 text-sm" placeholder="John"/></div>
                   <div><label className="block text-[11px] font-bold uppercase text-slate-500 mb-1.5">Last Name</label><input required name="lastName" value={formData.lastName} onChange={handleChange} type="text" className="w-full bg-white border border-slate-300 rounded-lg p-3 text-sm" placeholder="Doe"/></div>
                </div>
                <div><label className="block text-[11px] font-bold uppercase text-slate-500 mb-1.5">Email Address</label><input required name="email" value={formData.email} onChange={handleChange} type="email" className="w-full bg-white border border-slate-300 rounded-lg p-3 text-sm" placeholder="john@example.com"/></div>
                <div>
                   <label className="block text-[11px] font-bold uppercase text-slate-500 mb-2">Payment Method</label>
                   <div className="grid grid-cols-3 gap-3">
                      {['card', 'paypal', 'crypto'].map(m => (
                        <button key={m} type="button" onClick={() => setPaymentMethod(m)} className={`border font-bold text-xs py-3 rounded-lg capitalize transition ${paymentMethod === m ? 'border-slate-900 bg-slate-900 text-white' : 'border-slate-200 text-slate-500'}`}>{m}</button>
                      ))}
                   </div>
                </div>
                {paymentMethod === 'card' && (
                  <div className="bg-slate-50 p-4 rounded-lg border border-slate-200">
                    <div className="mb-3"><input required name="cardName" value={formData.cardName} onChange={handleChange} type="text" placeholder="Name on Card" className="w-full bg-white border border-slate-300 rounded p-2 text-sm"/></div>
                    <div className="mb-3"><input required name="cardNumber" value={formData.cardNumber} onChange={handleChange} type="text" placeholder="Card Number" className="w-full bg-white border border-slate-300 rounded p-2 text-sm"/></div>
                    <div className="grid grid-cols-2 gap-3">
                       <input required name="expiry" value={formData.expiry} onChange={handleChange} type="text" placeholder="MM/YY" className="w-full bg-white border border-slate-300 rounded p-2 text-sm"/>
                       <input required name="cvc" value={formData.cvc} onChange={handleChange} type="text" placeholder="CVC" className="w-full bg-white border border-slate-300 rounded p-2 text-sm"/>
                    </div>
                  </div>
                )}
                <button disabled={isProcessing} type="submit" className="w-full bg-green-600 hover:bg-green-700 text-white font-bold py-4 rounded-xl mt-6 flex justify-center items-center gap-3 text-lg">
                   {isProcessing ? "Processing..." : <>Complete Order <ArrowRight size={20}/></>}
                </button>
             </form>
          </div>
       </div>
    </div>
  );
};

// --- PAGES ---

// 1. ABOUT US PAGE
const AboutPage = ({ config }) => (
  <div className="max-w-4xl mx-auto px-4 py-20 animate-fade-in-up text-center">
    <div className="bg-slate-900 p-4 rounded-full w-20 h-20 mx-auto flex items-center justify-center mb-6 border border-slate-700"><Users size={32} className="text-green-500" /></div>
    <h1 className="text-4xl font-black text-white mb-8">About Us</h1>
    <div className="bg-slate-900/50 p-8 rounded-2xl border border-white/5 text-lg text-slate-300 leading-relaxed whitespace-pre-line">
      {config.aboutUs.content}
    </div>
  </div>
);

// 2. CONTACT US PAGE
const ContactPage = ({ onSendMessage }) => {
  const [form, setForm] = useState({ name: '', email: '', message: '' });
  const [submitted, setSubmitted] = useState(false);

  const handleSubmit = (e) => {
    e.preventDefault();
    onSendMessage({ ...form, date: new Date().toISOString(), read: false });
    setSubmitted(true);
  };

  if (submitted) return (
    <div className="max-w-md mx-auto px-4 py-20 text-center animate-fade-in">
       <div className="bg-green-500/20 p-6 rounded-full w-24 h-24 mx-auto flex items-center justify-center mb-6"><Check size={40} className="text-green-500" /></div>
       <h2 className="text-3xl font-bold text-white mb-2">Message Sent!</h2>
       <p className="text-slate-400">We will get back to you shortly.</p>
    </div>
  );

  return (
    <div className="max-w-xl mx-auto px-4 py-20 animate-fade-in-up">
      <h1 className="text-4xl font-black text-white mb-8 text-center">Contact Us</h1>
      <form onSubmit={handleSubmit} className="bg-slate-900 p-8 rounded-2xl border border-slate-800 space-y-6">
         <div>
            <label className="block text-xs font-bold text-slate-500 uppercase mb-2">Your Name</label>
            <input required value={form.name} onChange={e => setForm({...form, name: e.target.value})} className="w-full bg-slate-950 border border-slate-800 rounded-lg p-3 text-white focus:border-green-500 outline-none" placeholder="Enter name"/>
         </div>
         <div>
            <label className="block text-xs font-bold text-slate-500 uppercase mb-2">Email Address</label>
            <input required value={form.email} onChange={e => setForm({...form, email: e.target.value})} type="email" className="w-full bg-slate-950 border border-slate-800 rounded-lg p-3 text-white focus:border-green-500 outline-none" placeholder="Enter email"/>
         </div>
         <div>
            <label className="block text-xs font-bold text-slate-500 uppercase mb-2">Message</label>
            <textarea required value={form.message} onChange={e => setForm({...form, message: e.target.value})} className="w-full bg-slate-950 border border-slate-800 rounded-lg p-3 text-white focus:border-green-500 outline-none h-32" placeholder="How can we help?"/>
         </div>
         <button type="submit" className="w-full bg-green-600 hover:bg-green-500 text-white font-bold py-3 rounded-lg transition">Send Message</button>
      </form>
    </div>
  );
};

// 3. DOWNLOAD / ORDER TRACKING PAGE
const DownloadPage = ({ config, orders, currentUserEmail }) => {
  const [email, setEmail] = useState(currentUserEmail || '');
  const [verified, setVerified] = useState(!!currentUserEmail);
  const [error, setError] = useState('');

  const handleVerify = (e) => {
    e.preventDefault();
    const orderExists = orders.find(o => o.email.toLowerCase() === email.toLowerCase());
    if (orderExists) {
      setVerified(true);
      setError('');
    } else {
      setError('No active license found for this email. Please purchase first.');
    }
  };

  return (
    <div className="max-w-2xl mx-auto px-4 py-20 text-center animate-fade-in-up">
      {!verified ? (
        <div className="bg-slate-900 p-8 rounded-2xl border border-slate-800">
          <div className="bg-slate-800 w-16 h-16 rounded-full flex items-center justify-center mx-auto mb-6"><Lock className="text-slate-400"/></div>
          <h1 className="text-3xl font-bold text-white mb-2">Access Your Downloads</h1>
          <p className="text-slate-400 mb-8">Enter the email address you used during checkout to access your files.</p>
          <form onSubmit={handleVerify} className="max-w-sm mx-auto space-y-4">
             <input type="email" required value={email} onChange={e => setEmail(e.target.value)} className="w-full bg-slate-950 border border-slate-800 rounded-lg p-3 text-white text-center focus:border-green-500 outline-none" placeholder="name@example.com"/>
             <button type="submit" className="w-full bg-white text-black font-bold py-3 rounded-lg hover:bg-slate-200 transition">Verify Access</button>
             {error && <p className="text-red-400 text-sm mt-2">{error}</p>}
          </form>
        </div>
      ) : (
        <div className="bg-slate-900 p-8 rounded-2xl border border-slate-800 relative overflow-hidden">
           <div className="absolute top-0 left-0 w-full h-1 bg-green-500"></div>
           <div className="bg-green-500/20 w-20 h-20 rounded-full flex items-center justify-center mx-auto mb-6"><Check size={40} className="text-green-500"/></div>
           <h1 className="text-3xl font-bold text-white mb-2">Access Granted</h1>
           <p className="text-green-400 font-mono mb-8">License Verified for: {email}</p>
           
           <div className="bg-slate-950 p-6 rounded-xl border border-slate-800 text-left flex items-center justify-between group hover:border-green-500/50 transition">
              <div className="flex items-center gap-4">
                 <div className="bg-slate-800 p-3 rounded-lg"><FileText className="text-white"/></div>
                 <div>
                    <h3 className="font-bold text-white text-lg">{config.general.siteName} Bot</h3>
                    <p className="text-slate-500 text-sm">{config.downloads.version}</p>
                 </div>
              </div>
              <a href={config.downloads.fileLink} target="_blank" rel="noopener noreferrer" className="bg-green-600 hover:bg-green-500 text-white px-6 py-3 rounded-lg font-bold flex items-center gap-2 transition">
                 <Download size={18}/> Download
              </a>
           </div>
           <p className="text-slate-500 text-xs mt-6">Secure download link generated. Do not share this link.</p>
        </div>
      )}
    </div>
  );
};

// 4. MAIN LAYOUT (Navbar/Footer)
const Layout = ({ children, config, onNavigate, currentRoute, onBuy }) => {
  const getColor = () => {
    const c = config.general.primaryColor;
    return c === 'blue' ? 'text-blue-500' : c === 'purple' ? 'text-purple-500' : c === 'yellow' ? 'text-yellow-500' : 'text-green-500';
  };
  const themeText = getColor();

  return (
    <div className="min-h-screen bg-slate-950 text-white font-sans selection:bg-white selection:text-black flex flex-col">
       {/* CSS FOR MARQUEE */}
       <style>{`
         @keyframes marquee { 0% { transform: translateX(0); } 100% { transform: translateX(-50%); } }
         .animate-marquee { animation: marquee 30s linear infinite; }
       `}</style>

       {/* Top Bar */}
       {config.topBar.isEnabled && (
        <div className="bg-gradient-to-r from-red-900 via-red-800 to-red-900 text-center py-2 text-xs font-bold uppercase tracking-wider text-white shadow-xl relative z-50">
           <span className="relative z-10 flex justify-center items-center gap-2">
             <AlertCircle size={14} /> {config.topBar.text} <span className="bg-white text-red-900 px-1 rounded">{config.topBar.highlightText}</span>
           </span>
        </div>
      )}

      {/* TOP TICKER */}
      {config.ticker.isEnabled && (
        <div className="bg-slate-900 border-b border-white/10 py-2 overflow-hidden relative z-40">
          <div className="flex w-full">
            <div className={`whitespace-nowrap text-xs font-mono font-bold ${themeText} animate-marquee flex min-w-full items-center`}>
               <span className="mx-8 flex items-center gap-2"><Activity size={12}/> {config.ticker.text}</span>
               <span className="mx-8 flex items-center gap-2"><Activity size={12}/> {config.ticker.text}</span>
               <span className="mx-8 flex items-center gap-2"><Activity size={12}/> {config.ticker.text}</span>
            </div>
          </div>
        </div>
      )}

      {/* Navbar */}
      <nav className="sticky top-0 z-40 backdrop-blur-xl bg-slate-950/80 border-b border-white/5">
        <div className="max-w-7xl mx-auto px-6 py-4 flex justify-between items-center">
          <div onClick={() => onNavigate('home')} className="text-xl font-black tracking-tighter flex items-center gap-2 cursor-pointer hover:opacity-80">
            <Cpu className={themeText} />
            <span>{config.general.siteName}</span>
          </div>
          <div className="hidden md:flex items-center gap-6 text-sm font-medium text-slate-400">
             <button onClick={() => onNavigate('home')} className={`hover:text-white ${currentRoute === 'home' ? 'text-white' : ''}`}>Home</button>
             <button onClick={() => onNavigate('about')} className={`hover:text-white ${currentRoute === 'about' ? 'text-white' : ''}`}>About</button>
             <button onClick={() => onNavigate('contact')} className={`hover:text-white ${currentRoute === 'contact' ? 'text-white' : ''}`}>Contact</button>
             <button onClick={() => onNavigate('download')} className={`hover:text-white ${currentRoute === 'download' ? 'text-white' : ''}`}>Track Order</button>
          </div>
          <button onClick={onBuy} className="bg-green-600 hover:bg-green-500 text-white px-5 py-2 rounded-full font-bold text-xs transition shadow-lg shadow-green-900/20">Buy Now</button>
        </div>
      </nav>

      {/* Content */}
      <div className="flex-1 pb-10">{children}</div>

      {/* Footer */}
      <footer className="py-8 bg-slate-950 text-center text-slate-700 text-xs border-t border-white/5 select-none mt-auto">
        <div className="flex justify-center gap-4 mb-4">
          <button onClick={() => onNavigate('about')} className="hover:text-slate-400">About Us</button>
          <button onClick={() => onNavigate('contact')} className="hover:text-slate-400">Contact</button>
          <button onClick={() => onNavigate('download')} className="hover:text-slate-400">My Account</button>
        </div>
        <p onClick={() => onNavigate('admin-login')} className="hover:text-slate-500 transition cursor-default">
          &copy; 2025 {config.general.siteName} Technologies LTD.
        </p>
      </footer>

      {/* BOTTOM TICKER (Fixed) */}
      {config.ticker.isEnabled && (
        <div className="fixed bottom-0 left-0 w-full bg-slate-900 border-t border-white/10 py-2 z-50">
          <div className="flex w-full overflow-hidden">
            <div className={`whitespace-nowrap text-xs font-mono font-bold ${themeText} animate-marquee flex min-w-full items-center`}>
               <span className="mx-8 flex items-center gap-2"><Activity size={12}/> {config.ticker.text}</span>
               <span className="mx-8 flex items-center gap-2"><Activity size={12}/> {config.ticker.text}</span>
               <span className="mx-8 flex items-center gap-2"><Activity size={12}/> {config.ticker.text}</span>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

// 5. HOME PAGE
const HomePage = ({ config, onBuy }) => {
  const theme = config.general.primaryColor === 'green' ? 'text-green-500' : 'text-blue-500';
  const themeBg = config.general.primaryColor === 'green' ? 'bg-green-600' : 'bg-blue-600';

  return (
    <>
      {/* HERO */}
      <main className="max-w-7xl mx-auto px-4 py-16 text-center">
         <div className="inline-flex items-center gap-2 px-3 py-1.5 rounded text-[10px] font-mono font-bold border bg-slate-900 border-green-500/30 text-green-500 bg-opacity-20 tracking-widest uppercase mb-8">
            <div className="w-1.5 h-1.5 rounded-full bg-green-500 animate-pulse"></div> System Status: Operational
         </div>
         <h1 className="text-5xl md:text-7xl font-black leading-none tracking-tight text-white mb-6">
            {config.hero.headlinePrefix} <br />
            <span className="text-transparent bg-clip-text bg-gradient-to-r from-white to-slate-600">{config.hero.headlineSuffix}</span>
         </h1>
         <p className="text-lg text-slate-400 max-w-lg mx-auto leading-relaxed mb-8">{config.hero.subHeadline}</p>
         <button onClick={onBuy} className={`w-full sm:w-auto ${themeBg} text-white text-lg py-5 px-12 rounded-lg font-black uppercase tracking-wide shadow-2xl hover:-translate-y-1 transition-all flex items-center justify-center gap-3 mx-auto`}>
            {config.hero.ctaText} <ChevronRight strokeWidth={3} />
         </button>
         
         {config.hero.showChart && <LiveChart color={config.general.primaryColor} />}
      </main>

      {/* WHY CHOOSE US (New Section) */}
      <div className="bg-slate-900 border-y border-slate-800 py-20">
         <div className="max-w-6xl mx-auto px-4">
            <div className="text-center mb-12">
               <h2 className="text-3xl font-black text-white mb-4">Why Top Traders Choose Us</h2>
               <p className="text-slate-400">Outperform the market with institutional-grade technology.</p>
            </div>
            <div className="grid md:grid-cols-3 gap-8">
               {[
                 { title: "AI-Driven Precision", desc: "Our algorithm analyzes 50+ indicators in real-time to find high-probability entries.", icon: <Cpu size={32} className="text-blue-500" /> },
                 { title: "Zero-Risk Guard", desc: "Built-in equity protection ensures you never lose more than your predefined limit.", icon: <Shield size={32} className="text-green-500" /> },
                 { title: "24/7 Automation", desc: "Sleep while the bot trades for you. Works on VPS and desktop continuously.", icon: <Activity size={32} className="text-purple-500" /> },
               ].map((item, i) => (
                 <div key={i} className="bg-slate-950 p-8 rounded-2xl border border-slate-800 hover:border-slate-700 transition hover:-translate-y-1">
                    <div className="bg-slate-900 w-16 h-16 rounded-xl flex items-center justify-center mb-6">{item.icon}</div>
                    <h3 className="text-xl font-bold text-white mb-3">{item.title}</h3>
                    <p className="text-slate-400 text-sm leading-relaxed">{item.desc}</p>
                 </div>
               ))}
            </div>
         </div>
      </div>

      {/* REVIEWS (New Section) */}
      <div className="py-20 bg-slate-950">
         <div className="max-w-6xl mx-auto px-4">
            <h2 className="text-3xl font-black text-white text-center mb-12">Verified User Results</h2>
            <div className="grid md:grid-cols-3 gap-6">
               {[
                 { name: "Sarah J.", profit: "+$1,240", text: "I was skeptical at first, but the auto-stop loss saved me twice. Up 15% this week!", stars: 5 },
                 { name: "Michael C.", profit: "+$4,800", text: "Finally a bot that actually works on XAUUSD. Setup took literally 5 minutes.", stars: 5 },
                 { name: "David R.", profit: "+$890", text: "Good support team and the download was instant. Highly recommended.", stars: 4 },
               ].map((r, i) => (
                 <div key={i} className="bg-slate-900/50 p-6 rounded-2xl border border-white/5">
                    <div className="flex gap-1 text-yellow-500 mb-4">
                       {[...Array(r.stars)].map((_, i) => <Star key={i} size={16} fill="currentColor" />)}
                    </div>
                    <p className="text-slate-300 text-sm mb-6 italic">"{r.text}"</p>
                    <div className="flex items-center gap-3">
                       <div className="w-10 h-10 bg-slate-800 rounded-full flex items-center justify-center font-bold text-slate-500">{r.name.charAt(0)}</div>
                       <div>
                          <div className="text-white font-bold text-sm">{r.name}</div>
                          <div className="text-green-500 text-xs font-bold">Profit: {r.profit}</div>
                       </div>
                    </div>
                 </div>
               ))}
            </div>
         </div>
      </div>

      {/* PRICING */}
      <div className="py-20 bg-slate-900 border-t border-slate-800">
         <div className="max-w-3xl mx-auto px-4 text-center">
            <h2 className="text-3xl font-black text-white mb-8">Simple One-Time Pricing</h2>
            <div className="bg-slate-950 border border-slate-800 rounded-2xl p-1 relative">
               <div className={`absolute -top-4 left-1/2 -translate-x-1/2 ${themeBg} text-white px-4 py-1 rounded-full text-xs font-bold uppercase tracking-wider shadow-lg`}>
                 {config.product.discountBadge}
               </div>
               <div className="bg-slate-900 rounded-xl p-8 md:p-12">
                  <div className="flex justify-center items-end gap-2 mb-6">
                     <span className="text-5xl font-black text-white">{config.product.currency}{config.product.price}</span>
                     <span className="text-xl text-slate-500 line-through mb-2">{config.product.currency}{config.product.regularPrice}</span>
                  </div>
                  <button onClick={onBuy} className={`w-full ${themeBg} text-white py-4 rounded-lg font-bold uppercase tracking-widest shadow-lg hover:shadow-xl hover:scale-[1.02] transition-all mb-4`}>
                    Buy Now
                  </button>
               </div>
            </div>
         </div>
      </div>
    </>
  );
};

// --- ADMIN LOGIN ---
const AdminLoginPage = ({ onLogin }) => {
  const [pass, setPass] = useState('');
  const [err, setErr] = useState(false);
  
  const handleAuth = () => {
    if (pass === '12345') onLogin();
    else setErr(true);
  };

  return (
    <div className="min-h-screen bg-[#0f172a] flex items-center justify-center font-sans p-4">
      <div className="bg-white/5 backdrop-blur-xl border border-white/10 w-full max-w-sm p-8 rounded-2xl shadow-2xl">
        <div className="flex justify-center mb-6"><div className="bg-black p-4 rounded-xl shadow-xl"><Lock className="text-white"/></div></div>
        <h2 className="text-xl font-bold text-center text-white mb-8">Admin Access</h2>
        <input type="password" value={pass} onChange={e => {setPass(e.target.value); setErr(false);}} placeholder="Enter Pin" className="w-full bg-slate-900/50 border border-slate-700 rounded-lg p-3 text-white text-center tracking-widest mb-4 outline-none focus:border-white"/>
        {err && <p className="text-red-400 text-xs text-center mb-4">Access Denied. Incorrect Pin.</p>}
        <button onClick={handleAuth} className="w-full bg-white text-black font-bold py-3 rounded-lg transition hover:bg-gray-200">Authenticate</button>
      </div>
    </div>
  );
};

// --- ADMIN COMPONENTS (Moved outside to prevent re-renders) ---
const SidebarItem = ({ icon: I, label, id, active, onClick }) => (
  <button onClick={() => onClick(id)} className={`w-full flex items-center gap-3 px-4 py-3 rounded-lg transition ${active ? 'bg-slate-900 text-white' : 'text-slate-500 hover:bg-slate-100'}`}>
    <I size={18} /><span className="font-medium text-sm">{label}</span>
  </button>
);

const AdminInput = ({ label, value, onChange, type = "text" }) => (
  <div className="mb-5">
    <label className="block text-xs font-bold text-slate-500 mb-2 uppercase tracking-wider">{label}</label>
    {type === "textarea" ? <textarea value={value} onChange={onChange} className="w-full bg-slate-50 border border-slate-200 rounded-lg p-3 text-slate-900 focus:border-slate-400 focus:ring-0 outline-none transition h-24 font-medium resize-none" /> : <input type={type} value={value} onChange={onChange} className="w-full bg-slate-50 border border-slate-200 rounded-lg p-3 text-slate-900 focus:border-slate-400 focus:ring-0 outline-none transition font-medium" />}
  </div>
);

// --- ADMIN DASHBOARD ---
const AdminDashboard = ({ config, setConfig, orders, messages, onLogout }) => {
  const [activeTab, setActiveTab] = useState('orders');
  const [localConfig, setLocalConfig] = useState(config);
  const [saved, setSaved] = useState(false);

  const update = (sec, key, val) => setLocalConfig(p => ({...p, [sec]: {...p[sec], [key]: val}}));
  const save = () => { setConfig(localConfig); setSaved(true); setTimeout(() => setSaved(false), 2000); };

  return (
    <div className="min-h-screen bg-white text-slate-900 flex font-sans">
      <div className="w-64 border-r border-slate-200 flex flex-col">
        <div className="p-6 font-black text-lg flex items-center gap-2"><LayoutDashboard size={18}/> ADMIN</div>
        <nav className="p-4 space-y-1 flex-1 overflow-y-auto">
          <SidebarItem icon={List} label="Orders" id="orders" active={activeTab === 'orders'} onClick={setActiveTab} />
          <SidebarItem icon={MessageSquare} label="Inbox" id="messages" active={activeTab === 'messages'} onClick={setActiveTab} />
          <div className="my-2 border-t border-slate-100"></div>
          <SidebarItem icon={Settings} label="General Settings" id="general" active={activeTab === 'general'} onClick={setActiveTab} />
          <SidebarItem icon={AlertCircle} label="Flash Sale" id="topBar" active={activeTab === 'topBar'} onClick={setActiveTab} />
          <SidebarItem icon={Monitor} label="Hero Area" id="hero" active={activeTab === 'hero'} onClick={setActiveTab} />
          <SidebarItem icon={BarChart3} label="Live Stats" id="stats" active={activeTab === 'stats'} onClick={setActiveTab} />
          <SidebarItem icon={DollarSign} label="Pricing" id="product" active={activeTab === 'product'} onClick={setActiveTab} />
          <SidebarItem icon={Activity} label="Ticker" id="ticker" active={activeTab === 'ticker'} onClick={setActiveTab} />
          <div className="my-2 border-t border-slate-100"></div>
          <SidebarItem icon={Upload} label="Downloads" id="downloads" active={activeTab === 'downloads'} onClick={setActiveTab} />
          <SidebarItem icon={FileText} label="Content & Pages" id="content" active={activeTab === 'content'} onClick={setActiveTab} />
        </nav>
        <div className="p-4 border-t"><button onClick={onLogout} className="text-red-500 text-sm font-bold flex items-center gap-2"><LogOut size={16}/> Logout</button></div>
      </div>
      <div className="flex-1 flex flex-col h-screen overflow-hidden">
        <header className="border-b border-slate-200 py-4 px-8 flex justify-between items-center">
          <h1 className="text-xl font-bold capitalize">{activeTab.replace(/([A-Z])/g, ' $1')}</h1>
          <button onClick={save} className={`px-6 py-2 rounded-lg font-bold text-sm text-white transition ${saved ? 'bg-green-500' : 'bg-black'}`}>{saved ? 'Saved!' : 'Save Changes'}</button>
        </header>
        <div className="flex-1 overflow-y-auto p-8 bg-slate-50">
           <div className="max-w-4xl mx-auto bg-white rounded-xl shadow-sm border border-slate-200 p-8">
              {activeTab === 'orders' && (
                orders.length === 0 ? <p className="text-slate-400 text-center py-8">No orders yet.</p> :
                <table className="w-full text-left text-sm">
                  <thead className="border-b bg-slate-50 text-slate-500 uppercase text-xs"><tr><th className="p-3">User</th><th className="p-3">Email</th><th className="p-3">Status</th></tr></thead>
                  <tbody>{orders.map(o => <tr key={o.id} className="border-b"><td className="p-3 font-bold">{o.firstName}</td><td className="p-3">{o.email}</td><td className="p-3 text-green-600">Paid</td></tr>)}</tbody>
                </table>
              )}
              {activeTab === 'messages' && (
                 messages.length === 0 ? <p className="text-slate-400 text-center py-8">No messages yet.</p> :
                 messages.map((m, i) => (
                   <div key={i} className="border-b border-slate-100 py-4 last:border-0">
                     <div className="flex justify-between mb-1"><span className="font-bold">{m.name}</span><span className="text-xs text-slate-400">{new Date(m.date).toLocaleDateString()}</span></div>
                     <div className="text-xs text-blue-600 mb-2">{m.email}</div>
                     <p className="text-sm text-slate-600 bg-slate-50 p-3 rounded">{m.message}</p>
                   </div>
                 ))
              )}

              {activeTab === 'general' && (
                <>
                  <AdminInput label="Website Name" value={localConfig.general.siteName} onChange={(e) => update('general', 'siteName', e.target.value)} />
                  <div className="mb-6">
                    <label className="block text-xs font-bold text-slate-500 mb-3 uppercase tracking-wider">Brand Color</label>
                    <div className="flex gap-4">
                      {['green', 'blue', 'purple', 'yellow'].map(c => (
                        <button key={c} onClick={() => update('general', 'primaryColor', c)} className={`h-10 w-10 rounded-full border-2 flex items-center justify-center ${localConfig.general.primaryColor === c ? 'border-slate-900 ring-2 ring-slate-200' : 'border-transparent'}`} style={{ backgroundColor: c === 'green' ? '#22c55e' : c === 'blue' ? '#3b82f6' : c === 'purple' ? '#a855f7' : '#eab308' }} />
                      ))}
                    </div>
                  </div>
                </>
              )}

              {activeTab === 'topBar' && (
                <>
                  <div className="flex items-center justify-between mb-8 p-4 bg-slate-50 rounded-lg border border-slate-200">
                    <span className="font-bold text-slate-700 text-sm">Enable Flash Sale Bar</span>
                    <input type="checkbox" checked={localConfig.topBar.isEnabled} onChange={(e) => update('topBar', 'isEnabled', e.target.checked)} className="accent-slate-900 w-5 h-5" />
                  </div>
                  <AdminInput label="Sale Text (Left)" value={localConfig.topBar.text} onChange={(e) => update('topBar', 'text', e.target.value)} />
                  <AdminInput label="Highlight Text (Right/Box)" value={localConfig.topBar.highlightText} onChange={(e) => update('topBar', 'highlightText', e.target.value)} />
                  <AdminInput label="Timer (Minutes)" type="number" value={localConfig.topBar.timerMinutes} onChange={(e) => update('topBar', 'timerMinutes', e.target.value)} />
                </>
              )}

              {activeTab === 'hero' && (
                <>
                  <AdminInput label="Headline Line 1" value={localConfig.hero.headlinePrefix} onChange={(e) => update('hero', 'headlinePrefix', e.target.value)} />
                  <AdminInput label="Headline Line 2" value={localConfig.hero.headlineSuffix} onChange={(e) => update('hero', 'headlineSuffix', e.target.value)} />
                  <AdminInput label="Sub-Text" type="textarea" value={localConfig.hero.subHeadline} onChange={(e) => update('hero', 'subHeadline', e.target.value)} />
                  <AdminInput label="CTA Button" value={localConfig.hero.ctaText} onChange={(e) => update('hero', 'ctaText', e.target.value)} />
                </>
              )}

              {activeTab === 'stats' && (
                <>
                  <div className="flex items-center justify-between mb-8 p-4 bg-slate-50 rounded-lg border border-slate-200">
                    <span className="font-bold text-slate-700 text-sm">Show Stats Strip</span>
                    <input type="checkbox" checked={localConfig.stats.isEnabled} onChange={(e) => update('stats', 'isEnabled', e.target.checked)} className="accent-slate-900 w-5 h-5" />
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <AdminInput label="Total Profit (String)" value={localConfig.stats.totalProfit} onChange={(e) => update('stats', 'totalProfit', e.target.value)} />
                    <AdminInput label="Active Users" value={localConfig.stats.activeUsers} onChange={(e) => update('stats', 'activeUsers', e.target.value)} />
                    <AdminInput label="Win Rate Display" value={localConfig.stats.dailyWinRate} onChange={(e) => update('stats', 'dailyWinRate', e.target.value)} />
                  </div>
                </>
              )}

              {activeTab === 'product' && (
                <div className="grid grid-cols-2 gap-6">
                  <AdminInput label="Price" type="number" value={localConfig.product.price} onChange={(e) => update('product', 'price', e.target.value)} />
                  <AdminInput label="Regular Price" type="number" value={localConfig.product.regularPrice} onChange={(e) => update('product', 'regularPrice', e.target.value)} />
                  <AdminInput label="Currency" value={localConfig.product.currency} onChange={(e) => update('product', 'currency', e.target.value)} />
                  <AdminInput label="Badge Text" value={localConfig.product.discountBadge} onChange={(e) => update('product', 'discountBadge', e.target.value)} />
                </div>
              )}

              {activeTab === 'ticker' && (
                 <AdminInput label="Bottom Ticker Content" type="textarea" value={localConfig.ticker.text} onChange={(e) => update('ticker', 'text', e.target.value)} />
              )}

              {activeTab === 'downloads' && (
                <div>
                   <h3 className="font-bold mb-4">File Management</h3>
                   <div className="mb-4">
                      <label className="block text-xs font-bold uppercase text-slate-500 mb-2">Bot Download Link (URL)</label>
                      <input type="text" value={localConfig.downloads.fileLink} onChange={e => update('downloads', 'fileLink', e.target.value)} className="w-full border p-3 rounded text-sm"/>
                      <p className="text-xs text-slate-400 mt-1">Paste your Google Drive, Dropbox, or AWS link here.</p>
                   </div>
                   <div>
                      <label className="block text-xs font-bold uppercase text-slate-500 mb-2">Version Name</label>
                      <input type="text" value={localConfig.downloads.version} onChange={e => update('downloads', 'version', e.target.value)} className="w-full border p-3 rounded text-sm"/>
                   </div>
                </div>
              )}

              {activeTab === 'content' && (
                <div>
                  <h3 className="font-bold mb-4">Page Content</h3>
                  <label className="block text-xs font-bold uppercase text-slate-500 mb-2">About Us Text</label>
                  <textarea value={localConfig.aboutUs.content} onChange={e => update('aboutUs', 'content', e.target.value)} className="w-full border p-3 rounded text-sm h-40"/>
                </div>
              )}
           </div>
        </div>
      </div>
    </div>
  );
};

// --- MAIN ROUTER ---
export default function FullProjectApp() {
  const [route, setRoute] = useState('home'); 
  const [config, setConfig] = useState(initialSiteConfig);
  const [orders, setOrders] = useState([]);
  const [messages, setMessages] = useState([]);
  const [checkoutOpen, setCheckoutOpen] = useState(false);
  const [currentUserEmail, setCurrentUserEmail] = useState('');

  const handleOrder = (order) => {
    setOrders([order, ...orders]);
    setCurrentUserEmail(order.email);
    setRoute('download'); // Redirect to download page on success
  };

  const handleMessage = (msg) => {
    setMessages([msg, ...messages]);
  };

  const renderContent = () => {
    switch(route) {
      case 'home': return <HomePage config={config} onBuy={() => setCheckoutOpen(true)} />;
      case 'about': return <AboutPage config={config} />;
      case 'contact': return <ContactPage onSendMessage={handleMessage} />;
      case 'download': return <DownloadPage config={config} orders={orders} currentUserEmail={currentUserEmail} />;
      default: return <HomePage config={config} onBuy={() => setCheckoutOpen(true)} />;
    }
  };

  if (route === 'admin-login') return <AdminLoginPage onLogin={() => setRoute('admin-dashboard')} />;
  if (route === 'admin-dashboard') return <AdminDashboard config={config} setConfig={setConfig} orders={orders} messages={messages} onLogout={() => setRoute('home')} />;

  return (
    <>
      {checkoutOpen && <CheckoutModal config={config} onClose={() => setCheckoutOpen(false)} onOrderPlaced={handleOrder} />}
      <Layout config={config} onNavigate={setRoute} currentRoute={route} onBuy={() => setCheckoutOpen(true)}>
        {renderContent()}
      </Layout>
    </>
  );
}